<?php
namespace RawMaterial\Controllers;

use User;
use DateTime;
use Grocery_CRUD;
use Tendoo_Module;
use RawMaterial\Classes\Material;

class RecipeController extends Tendoo_Module
{
    public function recipeCrud( $page = '', $id = null )
    {
        $material       =   new Material;
        $materials      =   $material->getMaterials();
        $unitsGroups    =   $material->getUnitGroupWithEntities();

        if ( $page === 'add' ) {
            $this->Gui->set_title( __( 'Composing a recipe', 'raw-material' ) );
            return $this->load->module_view( 'raw-material', 'recipes.add-gui', compact( 'materials', 'unitsGroups' ), true );
        } else if ( $page === 'edit' ) {
            $recipe         =   $material->getRecipe( $id );

            if ( $recipe ) {
                $this->Gui->set_title( __( 'Edit a recipe', 'raw-material' ) );
                return $this->load->module_view( 'raw-material', 'recipes.add-gui', compact( 'materials', 'unitsGroups', 'recipe' ), true );
            } else {
                return show_error( __( 'Unable to find the requested recipe', 'raw-material' ) );
            }
        } else {
            $this->Gui->set_title( __( 'Available Recipes', 'raw-material' ) );
            $data[ 'crud_content' ]   =   $this->recipeCrudHeader();
            return $this->load->module_view( 'raw-material', 'common.gui', $data, true );
        }
    }

    public function recipeCrudHeader()
    {
        /**
		 * This feature is not more accessible on main site when
		 * multistore is enabled
		**/
		
		if( ( multistore_enabled() && ! is_multistore() ) ) {
			return show_error( __( 'This feature doesn\'t works on the warehouse.', 'raw-material' ) );
		}
        
        $crud = new Grocery_CRUD();

        $crud->set_theme('bootstrap');
        $crud->set_subject(__( 'Material Recipes', 'raw-material' ));
        $crud->set_table( $this->db->dbprefix( store_prefix() . 'gastro_materials_recipes' ) );
		
		$fields					=	array( 'NAME', 'DESCRIPTION', 'AUTHOR', 'DATE_CREATION', 'DATE_MOD' );
		$crud->columns( 'NAME', 'AUTHOR', 'DATE_MOD' );
        $crud->fields( $fields );
        
        $state = $crud->getState();

        /**
         * Callback to support date formating
         * @since 3.12.8
         */
        $crud->callback_column( 'DATE_CREATION', function( $date ) {
            $datetime   =    new DateTime( $date ); 
            return $datetime->format( store_option( 'nexo_datetime_format', 'Y-m-d h:i:s' ) );
        });
        
        $crud->callback_column( 'DATE_MOD', function( $date ) {
            $datetime   =    new DateTime( $date ); 
            return $datetime->format( store_option( 'nexo_datetime_format', 'Y-m-d h:i:s' ) );
        });

        $crud->display_as( 'NAME',              __( 'Unit Name', 'raw-material' ) );
        $crud->display_as( 'DESCRIPTION',       __( 'Descrpition', 'raw-material' ) );
        $crud->display_as( 'AUTHOR',            __( 'Author', 'raw-material' ) );
        $crud->display_as( 'DATE_CREATION',     __( 'Created On', 'raw-material' ) );
        $crud->display_as( 'DATE_MOD',          __( 'Modified On', 'raw-material' ) );

        $crud->callback_before_insert( function( $data ) {
            $data[ 'AUTHOR' ]           =   User::id();
            $data[ 'DATE_CREATION' ]    =   date_now();
            $data[ 'DATE_MOD' ]         =   date_now();
            return $data;
        });

        $crud->callback_before_update( function( $data ) {
            $data[ 'AUTHOR' ]           =   User::id();
            $data[ 'DATE_MOD' ]         =   date_now();
            return $data;
        });

        $crud->callback_before_delete( function( $id ) {
            nexo_availability_check(
                $id,
                [
                    [
                        'col'       =>  'REF_RECIPE',
                        'table'     =>  store_prefix() . 'nexo_articles'
                    ]
                ]
            );

            $material       =   new Material;
            $material->deleteRecipeIngredients( $id );
        });
        
        // XSS Cleaner
        $this->events->add_filter('grocery_callback_insert', array( $this->grocerycrudcleaner, 'xss_clean' ));
        $this->events->add_filter('grocery_callback_update', array( $this->grocerycrudcleaner, 'xss_clean' ));
        
        $crud->required_fields('NAME');

        $crud->change_field_type('DATE_CREATION', 'invisible');
        $crud->change_field_type('DATE_MOD', 'invisible');
        $crud->change_field_type('AUTHOR', 'invisible');

        $crud->set_relation('AUTHOR', 'aauth_users', 'name');
        
        $crud->unset_jquery();
        $output = $crud->render();
        
        foreach ($output->js_files as $files) {
            $this->enqueue->js(substr($files, 0, -3), '');
        }
        foreach ($output->css_files as $files) {
            $this->enqueue->css(substr($files, 0, -4), '');
        }
        
        return $output;
    }
}