<?php
namespace RawMaterial\Controllers;

use User;
use DateTime;
use Grocery_CRUD;
use Tendoo_Module;
use RawMaterial\Inc\Setup;
use RawMaterial\Classes\Material;

class DashboardController extends Tendoo_Module
{
    public function __construct()
    {
        parent::__construct();
    }

    public function recipeCrud()
    {
        $this->load->module_view( 'raw-material', 'recipe.gui' );
    }

    public function supplies()
    {
        $this->Gui->set_title( store_title( __( 'Raw Material Supplies', 'raw-material' ) ) );
        $this->load->module_view( 'raw-material', 'supplies.gui' );
    }

    public function units()
    {
        $data[ 'crud_content' ]    	=   $this->unitsCrudHead();
        $this->Gui->set_title( store_title( __( 'Raw Material Units', 'raw-material' ) ) );
        $this->load->module_view( 'raw-material', 'units.gui', $data );
    }

    public function unitGroups()
    {
        $data[ 'crud_content' ]    	=   $this->unitsGroupsCrudHead();
        $this->Gui->set_title( store_title( __( 'Raw Material Units Group', 'raw-material' ) ) );
        $this->load->module_view( 'raw-material', 'units.gui', $data );
    }

    public function unitsGroupsCrudHead()
    {
		/**
		 * This feature is not more accessible on main site when
		 * multistore is enabled
		**/
		
		if( ( multistore_enabled() && ! is_multistore() ) ) {
			return show_error( __( 'This feature doesn\'t works on the warehouse.', 'raw-material' ) );
		}
        
        $crud = new Grocery_CRUD();

        $crud->set_theme('bootstrap');
        $crud->set_subject(__('Material Units Groups', 'raw-material' ));
        $crud->set_table( $this->db->dbprefix( store_prefix() . 'gastro_materials_units_groups' ) );
		
		$fields					=	array( 'NAME', 'DESCRIPTION', 'AUTHOR', 'DATE_CREATION', 'DATE_MOD' );
		$crud->columns( 'NAME', 'AUTHOR', 'DATE_MOD' );
        $crud->fields( $fields );
        
        $state = $crud->getState();

        /**
         * Callback to support date formating
         * @since 3.12.8
         */
        $crud->callback_column( 'DATE_CREATION', function( $date ) {
            $datetime   =    new DateTime( $date ); 
            return $datetime->format( store_option( 'nexo_datetime_format', 'Y-m-d h:i:s' ) );
        });
        $crud->callback_column( 'DATE_MOD', function( $date ) {
            $datetime   =    new DateTime( $date ); 
            return $datetime->format( store_option( 'nexo_datetime_format', 'Y-m-d h:i:s' ) );
        });

        $crud->display_as( 'NAME',              __( 'Unit Name', 'raw-material' ) );
        $crud->display_as( 'DESCRIPTION',       __( 'Descrpition', 'raw-material' ) );
        $crud->display_as( 'AUTHOR',            __( 'Author', 'raw-material' ) );
        $crud->display_as( 'DATE_CREATION',     __( 'Created On', 'raw-material' ) );
        $crud->display_as( 'DATE_MOD',          __( 'Modified On', 'raw-material' ) );

        $crud->callback_before_insert( function( $data ) {
            $data[ 'AUTHOR' ]           =   User::id();
            $data[ 'DATE_CREATION' ]    =   date_now();
            $data[ 'DATE_MOD' ]         =   date_now();
            return $data;
        });

        $crud->callback_before_update( function( $data ) {
            $data[ 'AUTHOR' ]           =   User::id();
            $data[ 'DATE_MOD' ]         =   date_now();
            return $data;
        });

        $crud->callback_before_delete( function( $id ) {
            nexo_availability_check(
                $id,
                [
                    [
                        'col'       =>  'REF_UNIT_GROUP',
                        'table'     =>  store_prefix() . 'gastro_materials'
                    ]
                ]
            );

            $material   =   new Material;
            $material->deleteUnitGroupEntities( $id );
        });

        $crud->add_action(
			__( 'Add New Unit', 'raw-material'),
            '',
            '',
			'fa fa-list',
			[ $this, 'getLink' ]
        );

        $crud->add_action(
			__( 'See Created Units', 'raw-material'),
            '',
            '',
			'fa fa-eye',
			[ $this, 'getAllUnitLink' ]
        );
        
        // XSS Cleaner
        $this->events->add_filter('grocery_callback_insert', array( $this->grocerycrudcleaner, 'xss_clean' ));
        $this->events->add_filter('grocery_callback_update', array( $this->grocerycrudcleaner, 'xss_clean' ));
        
        $crud->required_fields('NAME');

        $crud->change_field_type('DATE_CREATION', 'invisible');
        $crud->change_field_type('DATE_MOD', 'invisible');
        $crud->change_field_type('AUTHOR', 'invisible');

        $crud->set_relation('AUTHOR', 'aauth_users', 'name');
        
        $crud->unset_jquery();
        $output = $crud->render();
        
        foreach ($output->js_files as $files) {
            $this->enqueue->js(substr($files, 0, -3), '');
        }
        foreach ($output->css_files as $files) {
            $this->enqueue->css(substr($files, 0, -4), '');
        }
        
        return $output;
    }

    public function getLink( $id ) 
    {
        return site_url(array( 'dashboard', store_slug(), 'nexo', 'raw-material', 'units', $id, 'add-unit' )) . '/';
    }

    public function getAllUnitLink( $id ) 
    {
        return site_url(array( 'dashboard', store_slug(), 'nexo', 'raw-material', 'units', $id )) . '/';
    }

    public function unitsCrudHead()
    {
		/**
		 * This feature is not more accessible on main site when
		 * multistore is enabled
		**/
		
		if( ( multistore_enabled() && ! is_multistore() ) ) {
			return show_error( __( 'This feature doesn\'t works on the warehouse.', 'raw-material' ) );
		}
        
        $crud = new Grocery_CRUD();

        $crud->set_theme('bootstrap');
        $crud->set_subject(__('Material Units', 'raw-material' ));
        $crud->set_table( $this->db->dbprefix( store_prefix() . 'gastro_materials_units' ) );
		
		$fields					=	array( 'NAME', 'REF_UNIT_GROUP', 'UNIT_NAME', 'BASE_UNIT', 'UNIT_VALUE', 'DESCRIPTION', 'AUTHOR', 'DATE_CREATION', 'DATE_MOD' );
		$crud->columns( 'NAME', 'UNIT_NAME', 'BASE_UNIT', 'UNIT_VALUE', 'REF_UNIT_GROUP', 'DESCRIPTION', 'DATE_MOD' );
        $crud->fields( $fields );
        
        $state = $crud->getState();
        
        /**
         * Callback to support date formating
         * @since 3.12.8
         */
        $crud->callback_column( 'DATE_CREATION', function( $date ) {
            $datetime   =    new DateTime( $date ); 
            return $datetime->format( store_option( 'nexo_datetime_format', 'Y-m-d h:i:s' ) );
        });
        $crud->callback_column( 'DATE_MOD', function( $date ) {
            $datetime   =    new DateTime( $date ); 
            return $datetime->format( store_option( 'nexo_datetime_format', 'Y-m-d h:i:s' ) );
        });
        
        $crud->callback_column( 'BASE_UNIT', function( $value ) {
            return ( bool ) $value ? __( 'Yes', 'raw-material' ) : __( 'No', 'raw-material' );
        });
        
        
        $crud->display_as( 'NAME',              __( 'Unit Name', 'raw-material' ) );
		$crud->display_as( 'UNIT_NAME',         __( 'Unit Code', 'raw-material'  ) );
        $crud->display_as( 'UNIT_VALUE',        __( 'Unit Value', 'raw-material' ) );
		$crud->display_as( 'BASE_UNIT',         __( 'Base Unit', 'raw-material'  ) );
		$crud->display_as( 'REF_UNIT_GROUP',    __( 'Unit Group', 'raw-material'  ) );
        $crud->display_as( 'DESCRIPTION',       __( 'Descrpition', 'raw-material' ) );
        $crud->display_as( 'AUTHOR',            __( 'Author', 'raw-material' ) );
        $crud->display_as( 'DATE_CREATION',     __( 'Created On', 'raw-material' ) );

        $crud->field_description( 'UNIT_VALUE', __( 'This value should be based on the base unit, if this entity is not a base unit. It should be considered as multiplicator others unit (not for the base unit). E.g if a cup is 2.5ml (base unit) a bottle could be 10 time a cup (10 x 2.5).', 'raw-material' ) );
        $crud->field_description( 'UNIT_NAME', __( 'For example : kg, grams, l, liter, pack, piece', 'raw-material' ) );
        $crud->field_description( 'BASE_UNIT', __( 'This entity will be used as the base reference for the assigned group.', 'raw-material' ) );
        $crud->field_description( 'REF_UNIT_GROUP', __( 'A unit entity need to be assigned to a unit group.', 'raw-material' ) );

        $crud->callback_before_insert( function( $data ) {
            $data[ 'AUTHOR' ]           =   User::id();
            $data[ 'DATE_CREATION' ]    =   date_now();
            $data[ 'DATE_MOD' ]         =   date_now();

            return $data;
        });

        $crud->callback_before_update( function( $data ) {
            $data[ 'AUTHOR' ]           =   User::id();
            $data[ 'DATE_MOD' ]         =   date_now();
            return $data;
        });

        $crud->callback_after_insert( function( $data, $id ) {
            $material                   =   new Material;

            if ( ( bool ) $data[ 'BASE_UNIT' ] === true ) {
                $material->switchBaseUnit( $id );
            }
        });

        $crud->callback_after_update( function( $data, $id ) {
            $material                   =   new Material;

            if ( ( bool ) $data[ 'BASE_UNIT' ] === true ) {
                $material->switchBaseUnit( $id );
            }
        });
        
        // XSS Cleaner
        $this->events->add_filter('grocery_callback_insert', array( $this->grocerycrudcleaner, 'xss_clean' ));
        $this->events->add_filter('grocery_callback_update', array( $this->grocerycrudcleaner, 'xss_clean' ));
        
        $crud->required_fields([ 
            'NAME',
            'UNIT_NAME',
            'REF_UNIT_GROUP',
            'BASE_UNIT',
        ]);

        $crud->change_field_type('DATE_CREATION', 'invisible');
        $crud->change_field_type('DATE_MOD', 'invisible');
        $crud->change_field_type('AUTHOR', 'invisible');
        $crud->change_field_type('BASE_UNIT', 'dropdown', [
            1   =>  __( 'Yes', 'raw-material' ),
            0   =>  __( 'No', 'raw-material' ),
        ]);

        $crud->set_relation('AUTHOR', 'aauth_users', 'name');
        $crud->set_relation('REF_UNIT_GROUP', store_prefix() . 'gastro_materials_units_groups', 'NAME' );
        
        $crud->unset_jquery();
        $output = $crud->render();
        
        foreach ($output->js_files as $files) {
            $this->enqueue->js(substr($files, 0, -3), '');
        }
        foreach ($output->css_files as $files) {
            $this->enqueue->css(substr($files, 0, -4), '');
        }
        
        return $output;
    }

    public function settings()
    {
        $this->Gui->set_title( __( 'Raw Material Settings', 'raw-material' ) );
        $this->load->module_view( 'raw-material', 'settings.general.gui' );
    }

    public function install()
    {
        $setup      =   new Setup;
        $setup->install( $this->db->dbprefix . store_prefix() );

        return redirect( urldecode( $this->input->get( 'ref' ) ) );
    }

    public function test()
    {
        $material   =   new Material;
        echo json_encode( $material->deleteProcurementEntries(1) );
    }

    public function materialReport()
    {
        $material   =   new Material;
        $materials  =   $material->getMaterials();
        $units      =   $material->getUnits();

        $this->Gui->set_title( store_title( __( 'Material Report', 'raw-material' ) ) );
        $this->load->module_view( 'raw-material', 'common.dom', [
            'dom'   =>  $this->load->module_view( 'raw-material', 'reports/materials/dom', compact( 'units', 'materials' ), true )
        ]);
    }

    public function recipesReport()
    {
        $this->load->module_model( 'nexo', 'NexoProducts', 'product_model' );

        $material   =   new Material;
        $materials  =   $material->getMaterials();
        $units      =   $material->getUnits();
        $products   =   $this->product_model->get();
        $recipes    =   $material->getRecipes();

        $this->Gui->set_title( store_title( __( 'Recipe Report', 'raw-material' ) ) );
        $this->load->module_view( 'raw-material', 'common.dom', [
            'dom'   =>  $this->load->module_view( 'raw-material', 'reports/recipes/dom', compact( 'units', 'materials', 'products', 'recipes' ), true )
        ]);
    }

    public function materialStock()
    {
        $material   =   new Material;
        $materials  =   collect( $material->getMaterials() )->mapWithKeys( function( $material ) {
            return [ $material[ 'ID' ] => $material ];
        });
        $units      =   collect( $material->getUnits() )->mapWithKeys( function( $unit ) {
            return [ $unit[ 'ID' ] => $unit ];
        });
        $groups     =   collect( $material->getGroups() )->mapWithKeys( function( $group ) {
            return [ $group[ 'ID' ] => $group ];   
        });

        $keyUnits       =   $units->keys()->toArray();
        $keyGroups      =   $groups->keys()->toArray();
        $keyMaterials   =   $materials->keys()->toArray();

        $stocks     =   collect( $material->getMaterialStocks() )->filter( function( $stock ) use ( $units, $groups, $keyUnits, $keyGroups, $keyMaterials ) {
            return in_array( $stock[ 'REF_UNIT' ], $keyUnits ) && in_array( $stock[ 'REF_MATERIAL' ], $keyMaterials );
        });

        $this->Gui->set_title( store_title( __( 'Material Stock Report', 'raw-material' ) ) );
        $this->load->module_view( 'raw-material', 'common.dom', [
            'dom'   =>  $this->load->module_view( 'raw-material', 'reports/material-stocks/dom', compact( 'units', 'materials', 'stocks', 'groups' ), true )
        ]);
    }
}