<?php

namespace App\Services;

use App\Classes\Hook;
use App\Models\Option;

class Options
{
    private $rawOptions = [];

    public string $tableName;

    /**
     * the option class can be constructed with the user id.
     * If the user is not provided, the general options are loaded instead.
     */
    public function __construct()
    {
        $this->tableName = (new Option)->getTable();
        $this->build();
    }

    /**
     * Will reset the default options
     *
     * @param array $options
     */
    public function setDefault($options = []): void
    {
        Option::truncate();

        $defaultProviderCategories = [
            "Produce Suppliers",
            "Meat & Seafood Suppliers",
            "Dairy Suppliers",
            "Beverage Suppliers",
            "Dry Goods & Staples",
            "Frozen Food Suppliers",
            "Health & Organic Product Suppliers",
            "Snacks & Confectionery",
            "Personal Care & Household Supplies",
            "Imported & Specialty Foods"
        ];
        $defaultProviderTypes = [
            "Wholesale Distributor",
            "Food Supplier",
            "Beverage Supplier",
            "Dairy Supplier",
            "Meat Supplier",
            "Produce Supplier",
            "Baker",
            "Frozen Food Supplier",
            "Organic Food Supplier",
            "Seafood Supplier",
            "Dry Goods Supplier",
            "Bulk Goods Supplier",
            "Snack Supplier",
            "Canned Goods Supplier",
            "Cleaning Supplies Supplier",
            "Packaging Supplier",
            "Health & Wellness Product Supplier",
            "Non-perishable Goods Supplier",
            "Wholesale Grocer",
            "Vendor",
            "Local Supplier"
        ];
        $defaultOptions = [
            'ns_registration_enabled' => 'no',
            'ns_store_name' => 'ARYTechPOS',
            'ns_pos_allow_decimal_quantities' => 'yes',
            'ns_pos_quick_product' => 'yes',
            'ns_pos_show_quantity' => 'yes',
            // 'ns_currency_precision' => 2,
            'ns_pos_hide_empty_categories' => 'yes',
            'ns_pos_unit_price_ediable' => 'yes',
            'ns_pos_order_types' => ['takeaway', 'delivery'],
            //Added by Taimoor - 03:59 PM 13/FEB/2025
            //For the currency settings:
            'ns_currency_symbol' => env('NS_CURRENCY_SYMBOL', default: 'Rs.'),
            'ns_currency_iso' => env('NS_CURRENCY_ISO', 'PKR'),
            'ns_currency_position' => env('NS_CURRENCY_POSITION', 'before'),
            'ns_currency_prefered' => env('NS_CURRENCY_PREFERED', 'iso'),
            'ns_currency_precision' => env('NS_CURRENCY_PRECISION', '0'),
            //Added by Taimoor - 11:51 AM 21/FEB/2025
            //For the vendor options
            'ns_options_provider_categories' => env('NS_PROVIDER_CATEGORIES', implode(',', $defaultProviderCategories)),
            'ns_options_provider_types' => env('NS_PROVIDER_TYPES', implode(',', $defaultProviderTypes)),
            'ns_options_provider_payment_modes' => ['advance', 'credit', 'cash_on_delivery', 'installments'],
            'ns_options_provider_payment_terms' => ['30_days', '60_days', '90_days', 'upon_receipt'],
            //For the procurements options
            //TODO: define these in model as const
            'ns_pos_unit_mode' => 'advanced',
            'ns_pos_procurement_flow' => 'direct',
            'ns_pos_product_creation' => 'advanced',
            //TODO: define these in model as const
            //For the feature/display pos screen
            'ns_pos_display_sale_screen' => 'vertical'
        ];


        $options = array_merge($defaultOptions, $options);

        foreach ($options as $key => $value) {
            $this->set($key, $value);
        }
    }

    /**
     * return option service
     *
     * @return object
     */
    public function option()
    {
        return Option::where('user_id', null);
    }

    /**
     * Build
     * Build option array
     *
     * @return void
     **/
    public function build()
    {
        $this->options = [];

        if (Helper::installed() && empty($this->rawOptions)) {
            $this->rawOptions = $this->option()
                ->get()
                ->mapWithKeys(function ($option) {
                    return [
                        $option->key => $option,
                    ];
                });
        }
    }

    /**
     * Set Option
     *
     * @param string key
     * @param any value
     * @param bool force set
     * @return void
     **/
    public function set($key, $value, $expiration = null)
    {
        /**
         * if an option has been found,
         * it will save the new value and update
         * the option object.
         */
        $foundOption = collect($this->rawOptions)->map(function ($option, $index) use ($value, $key, $expiration) {
            if ($key === $index) {
                $this->hasFound = true;

                $this->encodeOptionValue($option, $value);

                $option->expire_on = $expiration;

                /**
                 * this should be overridable
                 * from a user option or any
                 * extending this class
                 */
                $option = $this->beforeSave($option);
                $option->save();

                return $option;
            }

            return false;
        })
            ->filter();

        /**
         * if the option hasn't been found
         * it will create a new Option model
         * and store with, then save it on the option model
         */
        if ($foundOption->isEmpty()) {
            $option = new Option;
            $option->key = trim(strtolower($key));
            $option->array = false;

            $this->encodeOptionValue($option, $value);

            $option->expire_on = $expiration;

            /**
             * this should be overridable
             * from a user option or any
             * extending this class
             */
            $option = $this->beforeSave($option);
            $option->save();
        } else {
            $option = $foundOption->first();
        }

        /**
         * Let's save the new option
         */
        $this->rawOptions[$key] = $option;

        return $option;
    }

    /**
     * Encodes the value for the option before saving.
     */
    public function encodeOptionValue(Option $option, mixed $value): void
    {
        if (is_array($value)) {
            $option->array = true;
            $option->value = json_encode($value);
        } elseif (empty($value) && !(bool) preg_match('/[0-9]{1,}/', $value)) {
            $option->value = '';
        } else {
            $option->value = $value;
        }
    }

    /**
     * Sanitizes values before storing on the database.
     */
    public function beforeSave(Option $option)
    {
        /**
         * sanitizing input to remove
         * all script tags
         */
        $option->value = strip_tags($option->value);

        return $option;
    }

    /**
     * Get options
     **/
    public function get(?string $key = null, mixed $default = null)
    {
        if ($key === null) {
            return $this->rawOptions;
        }

        $filtredOptions = collect($this->rawOptions)->filter(function ($option) use ($key) {
            return is_array($key) ? in_array($option->key, $key) : $option->key === $key;
        });

        $options = $filtredOptions->map(function ($option) {
            $this->decodeOptionValue($option);

            return $option;
        });

        return match ($options->count()) {
            0 => $default,
            1 => $options->first()->value,
            default => $options->map(fn($option) => $option->value)->toArray()
        };
    }

    public function decodeOptionValue($option)
    {
        /**
         * We should'nt run this everytime we
         * try to pull an option from the database or from the array
         */
        if (!empty($option->value) && $option->isClean()) {
            if (is_string($option->value) && $option->array) {
                $json = json_decode($option->value, true);

                if (json_last_error() == JSON_ERROR_NONE) {
                    $option->value = $json;
                } else {
                    $option->value = null;
                }
            } elseif (!$option->array) {
                if (preg_match('/^[0-9]{1,}$/', $option->value)) {
                    $option->value = (int) $option->value;
                } elseif (preg_match('/^[0-9]{1,}\.[0-9]{1,}$/', $option->value)) {
                    $option->value = (float) $option->value;
                } else {
                    $option->value = $option->value;
                }
            }
        }
    }

    /**
     * Delete an option using a specific key.
     **/
    public function delete(string $key): void
    {
        $this->rawOptions = collect($this->rawOptions)->filter(function (Option $option) use ($key) {
            if ($option->key === $key) {
                $option->delete();

                return false;
            }

            return true;
        });
    }

    /**
     * It only returns the order process status
     *
     * @param string
     * @return string
     */

    public static function returnRefinedOptionsByKey($key = ''): array
    {
        if (empty($key)) {
            return [];
        }
        $optionsService = new Options();
        $options = $optionsService->get($key);
        if (!is_array($options)) {
            $options = explode(',', $options);
        }
        $refinedOptions = [];
        if (empty($options))
            return [];
        foreach ($options as $option) {
            $option = trim($option);
            $refinedOptions[] = array(
                'identifier' => str_replace(' ', '_', $option),
                'label' => ucwords(str_replace('_', ' ', $option))
            );
        }
        return $refinedOptions;
    }

    public static function returnLabelledOptions($filter = '', $options = []): array
    {
        if (empty($filter)) {
            return [];
        }
        if (empty($options)) {
            return [];
        }
        return Hook::filter($filter, collect($options)->mapWithKeys(function ($option) {
            return [
                $option['identifier'] => $option['label'],
            ];
        })->toArray());
    }

}
