<?php

namespace App\Crud;

use App\Models\Brand;
use App\Services\CrudEntry;
use App\Services\CrudService;
use App\Services\BrandService;
use App\Services\UsersService;
use App\Services\Helper;
use Illuminate\Http\Request;
use TorMorten\Eventy\Facades\Events as Hook;
use App\Classes\CrudForm;
use App\Classes\FormInput;
class BrandCrud extends CrudService
{
    /**
     * Define the autoload status
     */
    const AUTOLOAD = true;

    /**
     * Define the identifier
     */
    const IDENTIFIER = 'ns.brands';

    /**
     * define the base table
     */
    protected $table = 'nexopos_brands';

    /**
     * base route name
     */
    protected $mainRoute = 'ns.brands';

    /**
     * Define namespace
     *
     * @param  string
     */
    protected $namespace = 'ns.brands';

    /**
     * Model Used
     */
    protected $model = Brand::class;

    /**
     * Adding relation
     */
    public $relations = [
        ['nexopos_users', 'nexopos_users.id', '=', 'nexopos_brands.author']
    ];
    /**
     * Pick
     * Restrict columns you retrieve from relation.
     * Should be an array of associative keys, where
     * keys are either the related table or alias name.
     * Example : [
     *      'user'  =>  [ 'username' ], // here the relation on the table nexopos_users is using "user" as an alias
     * ]
     */
    /**
     * Define where statement
     *
     * @var array
     **/
    protected $listWhere = [];

    /**
     * Define where in statement
     *
     * @var array
     */
    protected $whereIn = [];

    /**
     * Fields which will be filled during post/put
     */
    public $fillable = [];

    /**
     * Define permissions
     *
     * @param  array
     */
    protected $permissions = [
        'create' => 'nexopos.create.brands',
        'read' => 'nexopos.read.brands',
        'update' => 'nexopos.update.brands',
        'delete' => 'nexopos.delete.brands',
    ];

    /**
     * Define Constructor
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Return the label used for the crud
     * instance
     *
     * @return array
     **/
    public function getLabels()
    {
        return [
            'list_title' => __('Brands List'),
            'list_description' => __('Display all brands.'),
            'no_entry' => __('No brands has been registered'),
            'create_new' => __('Add a new brand'),
            'create_title' => __('Create a new brand'),
            'create_description' => __('Register a new brand and save it.'),
            'edit_title' => __('Edit brand'),
            'edit_description' => __('Modify  Brand.'),
            'back_to_list' => __('Return to Brands'),
        ];
    }

    /**
     * Check whether a feature is enabled
     *
     **/
    public function isEnabled($feature): bool
    {
        return false; // by default
    }

    /**
     * Fields
     *
     * @param  object/null
     * @return array of field
     */
    public function getForm($entry = null)
{
    return CrudForm::form(
        main: FormInput::text(
            label: __('Name'),
            name: 'name',
            value: $entry->name ?? '',
            description: __('Provide a name to the resource.'),
            validation: 'required',
        ),
        tabs: CrudForm::tabs(
            CrudForm::tab(
                identifier: 'general',
                label: __('General'),
                fields: CrudForm::fields(
                    FormInput::switch(
                        name: 'active',
                        options: Helper::kvToJsOptions([__('No'), __('Yes')]),
                        label: __('Active'),
                        description: __('Define whether the brand is active or not.'),
                        value: ($entry !== null && $entry->active ? 1 : 0) ?? 0,
                    ),
                    FormInput::text(
                        name: 'report_name_placeholder',
                        label: __('Report Name Placeholder'),
                        description: __('Provide name that can be used for report names.'),
                        value: $entry->report_name_placeholder ?? ''
                    ),
                    FormInput::text(
                        name: 'email',
                        label: __('Email'),
                        description: __('Provide the brand email. Might be used to send automated email.'),
                        value: $entry->email ?? ''
                    ),
                    FormInput::text(
                        name: 'phone',
                        label: __('Phone'),
                        description: __('Contact phone number for the brand. Might be used to send automated SMS notifications.'),
                        value: $entry->phone ?? ''
                    ),
                    FormInput::text(
                        name: 'address',
                        label: __('Address'),
                        description: __('Address of the Brand.'),
                        value: $entry->address ?? ''
                    ),
                    FormInput::text(
                        name: 'website',
                        label: __('Website'),
                        description: __('Website of the Brand.'),
                        value: $entry->website ?? ''
                    )
                )
            )
        )
    );
}


    /**
     * Filter POST input fields
     *
     * @param  array of fields
     * @return array of fields
     */
    public function filterPostInputs($inputs)
    {
        return $inputs;
    }

    /**
     * Filter PUT input fields
     *
     * @param  array of fields
     * @return array of fields
     */
    public function filterPutInputs($inputs, Brand $entry)
    {
        return $inputs;
    }

    /**
     * Before saving a record
     *
     * @param  Request $request
     * @return void
     */
    public function beforePost($request)
    {
        $this->allowedTo('create');

        return $request;
    }

    /**
     * After saving a record
     *
     * @param  Request $request
     * @return void
     */
    public function afterPost($request, Brand $entry)
    {
        return $request;
    }

    /**
     * get
     *
     * @param  string
     * @return mixed
     */
    public function get($param)
    {
        switch ($param) {
            case 'model':
                return $this->model;
                break;
        }
    }

    /**
     * Before updating a record
     *
     * @param Request $request
     * @param  object entry
     * @return void
     */
    public function beforePut($request, $entry)
    {
        $this->allowedTo('update');

        return $request;
    }

    /**
     * After updating a record
     *
     * @param Request $request
     * @param  object entry
     * @return void
     */
    public function afterPut($request, $entry)
    {
        return $request;
    }

    /**
     * Before Delete
     *
     * @return void
     */
    public function beforeDelete($namespace, $id, $model)
    {
        if ($namespace == 'ns.brands') {
            $this->allowedTo('delete');
        }
    }

    /**
     * Define Columns
     */
    public function getColumns(): array
    {
        return [
            'name' => [
                'label' => __('Name'),
                '$direction' => '',
                '$sort' => false,
            ],
            'active' => [
                'label' => __('Active'),
                '$direction' => '',
                '$sort' => false,
            ],
            'email' => [
                'label' => __('Email'),
                '$direction' => '',
                '$sort' => false,
            ],
            'phone' => [
                'label' => __('Phone'),
                '$direction' => '',
                '$sort' => false,
            ],
            'nexopos_users_username' => [
                'label' => __('Author'),
                '$direction' => '',
                '$sort' => false,
            ],
            'created_at' => [
                'label' => __('Created At'),
                '$direction' => '',
                '$sort' => false,
            ],
        ];
    }

    /**
     * Define actions
     */
    public function setActions(CrudEntry $entry): CrudEntry
    {
        $entry->phone = $entry->phone ?? __('N/A');
        $entry->email = $entry->email ?? __('N/A');
        $entry->active = $entry->active ? __('Yes') : __('No');
        // $entry->amount_due = ns()->currency->define($entry->amount_due)->format();
        // $entry->amount_paid = ns()->currency->define($entry->amount_paid)->format();

        // Snippet 1
        $entry->action(
            identifier: 'edit',
            label: __('Edit'),
            type: 'GOTO',
            url: ns()->url('/dashboard/' . 'brands' . '/edit/' . $entry->id)
        );

        // Snippet 2
        // $entry->action(
        //     identifier: 'see-procurements',
        //     label: __('See Procurements'),
        //     type: 'GOTO',
        //     url: ns()->url('/dashboard/' . 'brands/' . $entry->id . '/procurements/')
        // );

        // Snippet 3
        // $entry->action(
        //     identifier: 'see-products',
        //     label: __('See Products'),
        //     type: 'GOTO',
        //     url: ns()->url('/dashboard/' . 'brands/' . $entry->id . '/products/')
        // );

        // Snippet 4
        $entry->action(
            identifier: 'delete',
            label: __('Delete'),
            type: 'DELETE',
            url: ns()->url('/api/crud/ns.brands/' . $entry->id),
            confirm: [
                'message' => __('Would you like to delete this ?'),
            ]
        );

        return $entry;
    }

    /**
     * Bulk Delete Action
     *
     * @param    object Request with object
     * @return  false/array
     */
    public function bulkAction(Request $request)
    {
        /**
         * Deleting licence is only allowed for admin
         * and supervisor.
         */
        $user = app()->make(UsersService::class);
        if (!$user->is(['admin', 'supervisor'])) {
            return response()->json([
                'status' => 'error',
                'message' => __('You\'re not allowed to do this operation'),
            ], 403);
        }

        if ($request->input('action') == 'delete_selected') {
            $status = [
                'success' => 0,
                'error' => 0,
            ];

            foreach ($request->input('entries') as $id) {
                $entity = $this->model::find($id);
                if ($entity instanceof Brand) {
                    $entity->delete();
                    $status['success']++;
                } else {
                    $status['error']++;
                }
            }

            return $status;
        }

        return Hook::filter($this->namespace . '-catch-action', false, $request);
    }

    /**
     * get Links
     *
     * @return array of links
     */
    public function getLinks(): array
    {
        return [
            'list' => ns()->url('dashboard/' . 'brands'),
            'create' => ns()->url('dashboard/' . 'brands/create'),
            'edit' => ns()->url('dashboard/' . 'brands/edit/'),
            'post' => ns()->url('api/crud/' . 'ns.brands'),
            'put' => ns()->url('api/crud/' . 'ns.brands/{id}' . ''),
        ];
    }

    /**
     * Get Bulk actions
     *
     * @return array of actions
     **/
    public function getBulkActions(): array
    {
        return Hook::filter($this->namespace . '-bulk', [
            [
                'label' => __('Delete Selected Groups'),
                'identifier' => 'delete_selected',
                'url' => ns()->route('ns.api.crud-bulk-actions', [
                    'namespace' => $this->namespace,
                ]),
            ],
        ]);
    }

    /**
     * get exports
     *
     * @return array of export formats
     **/
    public function getExports()
    {
        return [];
    }
}
